
// ============================================================
// 🌿 WELLNESS DYNAMIC LOADING FUNCTION
// Purpose: Load wellness mega menu content dynamically
// File: To be added to assets/js/main.js or mega-menu.js
// ============================================================

// Global variable to track if wellness content is loaded
let wellnessContentLoaded = false;

/**
 * Load Wellness Content Dynamically
 * Purpose: Fetch and inject wellness mega menu content
 * Trigger: When Wellness & Therapy tab is clicked
 */
async function loadWellnessContent() {
    console.log('🌿 WELLNESS LOADING: Dynamic loading initiated...');

    const loadingIndicator = document.getElementById('wellness-loading');
    const dynamicContainer = document.getElementById('wellness-dynamic-content');

    // Show loading indicator
    if (loadingIndicator) {
        loadingIndicator.style.display = 'flex';
        console.log('⏳ WELLNESS LOADING: Loading indicator displayed');
    }

    // Check if content is already loaded
    if (wellnessContentLoaded) {
        console.log('✅ WELLNESS CACHE: Content already loaded, skipping fetch');
        hideLoadingIndicator();
        return;
    }

    try {
        // Fetch wellness content from separate file
        console.log('📡 WELLNESS FETCH: Requesting partials/_wellness-menu-content.html');

        const response = await fetch('partials/_wellness-menu-content.html');

        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }

        const wellnessHTML = await response.text();
        console.log('✅ WELLNESS FETCH: Content retrieved successfully');
        console.log('📊 WELLNESS SIZE:', wellnessHTML.length, 'characters loaded');

        // Inject content into dynamic container
        if (dynamicContainer) {
            dynamicContainer.innerHTML = wellnessHTML;
            dynamicContainer.style.border = 'none';
            dynamicContainer.style.background = 'transparent';
            dynamicContainer.style.padding = '0';

            console.log('🎯 WELLNESS INJECT: Content injected into dynamic container');
        }

        // Mark as loaded
        wellnessContentLoaded = true;

        // Hide loading indicator after short delay for UX
        setTimeout(() => {
            hideLoadingIndicator();
            console.log('🚀 WELLNESS COMPLETE: Dynamic loading completed successfully');
        }, 1000);

        // Initialize any wellness-specific functionality
        initializeWellnessFeatures();

    } catch (error) {
        console.error('❌ WELLNESS ERROR: Failed to load wellness content:', error);

        // Show error message to user
        if (dynamicContainer) {
            dynamicContainer.innerHTML = `
                <div class="wellness-error-message">
                    <i class="fas fa-exclamation-triangle"></i>
                    <h4>Unable to Load Wellness Content</h4>
                    <p>Please try refreshing the page or contact support if the issue persists.</p>
                    <button onclick="loadWellnessContent()" class="retry-btn">
                        <i class="fas fa-redo"></i> Try Again
                    </button>
                </div>
            `;
        }

        hideLoadingIndicator();
    }
}

/**
 * Hide Loading Indicator
 * Purpose: Hide the wellness loading spinner
 */
function hideLoadingIndicator() {
    const loadingIndicator = document.getElementById('wellness-loading');
    if (loadingIndicator) {
        loadingIndicator.style.display = 'none';
        console.log('✅ WELLNESS UI: Loading indicator hidden');
    }
}

/**
 * Initialize Wellness Features
 * Purpose: Set up wellness-specific functionality after content loads
 */
function initializeWellnessFeatures() {
    console.log('🔧 WELLNESS INIT: Initializing wellness-specific features...');

    // Add click tracking for dynamically loaded wellness items
    const wellnessItems = document.querySelectorAll('.wellness-item');
    wellnessItems.forEach(item => {
        item.addEventListener('click', function() {
            const itemText = this.textContent.trim();
            console.log(`🛒 WELLNESS DYNAMIC CLICK: ${itemText} selected`);
        });
    });

    // Add click tracking for wellness CTA buttons
    const wellnessCTAs = document.querySelectorAll('.wellness-cta-btn');
    wellnessCTAs.forEach(btn => {
        btn.addEventListener('click', function() {
            const btnText = this.textContent.trim();
            console.log(`🎯 WELLNESS DYNAMIC CTA: ${btnText} clicked`);
        });
    });

    // Add hover effects for wellness categories
    const wellnessCards = document.querySelectorAll('.wellness-category-card');
    wellnessCards.forEach(card => {
        card.addEventListener('mouseenter', function() {
            console.log('🖱️ WELLNESS HOVER: Category card hovered');
        });
    });

    console.log('✅ WELLNESS INIT: All wellness features initialized');
}

/**
 * Preload Wellness Content (Optional)
 * Purpose: Preload wellness content in background for faster access
 * Call this on page load if desired
 */
function preloadWellnessContent() {
    console.log('🚀 WELLNESS PRELOAD: Background preloading initiated...');

    // Preload after a delay to not impact initial page load
    setTimeout(() => {
        if (!wellnessContentLoaded) {
            loadWellnessContent();
        }
    }, 3000); // 3 second delay
}

// ============================================================
// 🎯 WELLNESS ERROR HANDLING STYLES
// Purpose: Styling for error messages
// ============================================================
const wellnessErrorStyles = `
<style>
.wellness-error-message {
    text-align: center;
    padding: 40px 20px;
    background: #fff3cd;
    border: 1px solid #ffeaa7;
    border-radius: 10px;
    color: #856404;
}

.wellness-error-message i {
    font-size: 48px;
    color: #f39c12;
    margin-bottom: 15px;
}

.wellness-error-message h4 {
    color: #856404;
    font-size: 18px;
    margin-bottom: 10px;
}

.wellness-error-message p {
    color: #856404;
    margin-bottom: 20px;
}

.retry-btn {
    background: #4baf47;
    color: white;
    border: none;
    padding: 10px 20px;
    border-radius: 5px;
    cursor: pointer;
    font-weight: 600;
    transition: background 0.3s ease;
}

.retry-btn:hover {
    background: #12410d;
}
</style>
`;

// Inject error styles into document head
if (!document.getElementById('wellness-error-styles')) {
    const styleElement = document.createElement('style');
    styleElement.id = 'wellness-error-styles';
    styleElement.innerHTML = wellnessErrorStyles.replace('<style>', '').replace('</style>', '');
    document.head.appendChild(styleElement);
}

// ============================================================
// 🚀 WELLNESS PERFORMANCE MONITORING
// Purpose: Track loading performance
// ============================================================
function trackWellnessPerformance() {
    const startTime = performance.now();

    return {
        end: function() {
            const endTime = performance.now();
            const loadTime = (endTime - startTime).toFixed(2);
            console.log(`⏱️ WELLNESS PERFORMANCE: Content loaded in ${loadTime}ms`);
            return loadTime;
        }
    };
}

console.log('🌿 WELLNESS DYNAMIC LOADER: All functions ready and available');
